#include <float.h>
#include <math.h>
#include <assert.h>

#include <maya/MPxFileTranslator.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnTransform.h>
#include <maya/MPoint.h>
#include <maya/MBoundingBox.h>
#include <maya/MSelectionList.h>
#include <maya/MGlobal.h>
#include <maya/MTime.h>
#include <maya/MMatrix.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MDagPathArray.h>
#include <maya/MItGeometry.h>
#include <maya/MFloatArray.h>
#include <maya/MFnIkHandle.h>
#include <maya/MItDependencyGraph.h>
#include <maya/MFnAnimCurve.h>
#include <maya/MAnimControl.h>

#include <MDt.h>
#include <MDtExt.h>

#include <rpmorph.h>
#include <rpmatfx.h>

#include "rwcommon.h"
#include "material.h"
#include "remapper.h"
#include "querymel.h"
#include "node.h"
#include "blind.h"
#include "global.h"

/* sceneNode member functions */

sceneNode::sceneNode(int mdtNodeIndex, int numberChildren, bool processSkin)
{
    char *shapeName;

    mdtIndex                = mdtNodeIndex;
    parentMdtIndex          = DtShapeGetParentID(mdtIndex);
    numChildren             = numberChildren;
    frame                   = NULL;
    atomic                  = NULL;
    objectOffset.x          = 0.0f;
    objectOffset.y          = 0.0f;
    objectOffset.z          = 0.0f;
    objectScale.x           = 1.0f;
    objectScale.y           = 1.0f;
    objectScale.z           = 1.0f;

    numRemappedVerts        = 0;
    numRemappedFaces        = 0;
    remappedFaces           = NULL;
    remappedFaceMaterials   = NULL;
    remappedPositions       = NULL;
    remappedNormals         = NULL;
    remappedPreLightColors  = NULL;
    remappedTexCoords       = NULL;
    unmappedVertexIndices   = NULL;
    skinVertexIndices       = NULL;
    skinMatrixWeights       = NULL;
    vertexMap               = NULL;

    /*
        Reset the current frame to the animation start - it may have been
        changed when processing animation on previous frames in the hierarchy.
    */
    MAnimControl::setCurrentTime(MTime(globalData->m_animStart, MTime::uiUnit()));

    numRemappedVertsAfterMerge = 0;
    numRemappedFacesAfterMerge = 0;
    
    /* Get this nodes name */
    DtShapeGetName(mdtIndex, &shapeName);
    name = strdup(shapeName);

    /* Store a dag path to this node */
    DtExt_ShapeGetDagPath(mdtNodeIndex, dagPath);

    /* an MObject */
    DtExt_ShapeGetShapeNode(mdtIndex, object);
    
    /* Check if the node is a skin or a joint */
    hasSkinInfluences   = false;
    isSkinned           = false;
    checkSkinning();

    exportRpHAnim       = false;
    if (globalData->m_exportRpHAnim)
    {
        if (globalData->m_rpHAnimSubHierarchy)
        {
            exportRpHAnim = globalData->underSelectedMdtShape;
        }
        else
        {
            exportRpHAnim = true;
        }
    }
    
    /* Calculate the polygon count in the node */
    numPolys            = 0;
    calculatePolyCount();

    /* Decide the type of node */
    calculateNodeType();
    
    /* If we're verbose then output a newline to make things a little clearer */
    if (globalData->m_verbose)
    {
        printf("\n");
    }

    /* Output a node description */
    switch (type)
    {
        case geometryNode:
            if (isSkinned == true)
            {
                printf("Processing skinned geometry node named : %s\n", name);
            }
            else
            {
                printf("Processing geometry node named : %s\n", name);
            }
            break;
        case jointNode:
            printf("Processing joint node named : %s\n", name);
            break;
        case unknownNode:
            printf("Processing unknown node named : %s\n", name);
            break;
    }

    /* Decide if the node is textured */
    isTextured = false;
    checkTexturing();

    /* Calculate if this node has morph targets */
    calculateMorphKeys();

    /* Check if the lighting flag has been set on this shape */
    isLit = false;
    if (getBoolTagFromShape(mdtIndex, "RwLightFlag", &isLit) == false)
    {
        isLit = true;
    }

    /* And the merge tag */
    if (getIntTagFromShape(mdtIndex, "RwMergeGroup", &mergeGroup) == false)
    {
        mergeGroup = -1;
    }

    /* Store the modelling matrix of the node */
    frame = RwFrameCreate();
    if (isSkinned == true)
    {
        computeSkinnedRwMatrix(RwFrameGetMatrix(frame));
    }
    else
    {
        computeRwMatrix(RwFrameGetMatrix(frame), true);
    }

    /* Output the modelling matrix if we're being verbose */
    if (globalData->m_verbose)
    {
        RwMatrix *matrix;

        matrix = RwFrameGetMatrix(frame);
        
        printf("Object matrix:\n");
        printf("Right: %3.3f %3.3f %3.3f\n", matrix->right.x, matrix->right.y, matrix->right.z);
        printf("Up   : %3.3f %3.3f %3.3f\n", matrix->up.x, matrix->up.y, matrix->up.z);
        printf("At   : %3.3f %3.3f %3.3f\n", matrix->at.x, matrix->at.y, matrix->at.z);
        printf("Pos  : %3.3f %3.3f %3.3f\n", matrix->pos.x, matrix->pos.y, matrix->pos.z);
    }
    
    /* Process any animation present on the node */
    if (globalData->m_exportRpAnim ||
        globalData->m_exportRpSkin ||
        globalData->m_exportRpHAnim)
    {
        computeAnimation();
    }

    /* Process any blind data attached to the object */
    mayaBlindData = new blindData(object);

    /*
        If this node is skinned and we've been told not to
        process skins then bail out.
    */
    if ((isSkinned == true) && (processSkin == false))
    {
        return;
    }

    /* If we're not going to save a dff then ignore any geometry */
    if (globalData->m_cullDff)
    {
        return;
    }

    /* Remap the node geometry (if any) and store a copy */
    calculateRemappedGeometry();
}

sceneNode::~sceneNode()
{
    int i;

    if (name != NULL)
    {
        free(name);
    }

    if (remappedFaces != NULL)
    {
        free (remappedFaces);
    }
    if (remappedFaceMaterials != NULL)
    {
        free (remappedFaceMaterials);
    }
    
    if (remappedPositions != NULL)
    {
        for (i = 0; i < morphTargetKeys.length(); i++)
        {
            free (remappedPositions[i]);
        }
        free (remappedPositions);
    }
   
    if (remappedNormals != NULL)
    {
        for (i = 0; i < morphTargetKeys.length(); i++)
        {
            free (remappedNormals[i]);
        }
        free (remappedNormals);
    }

    if (remappedPreLightColors != NULL)
    {
        free (remappedPreLightColors);
    }

    if (remappedTexCoords != NULL)
    {
        free (remappedTexCoords);
    }

    if (unmappedVertexIndices != NULL)
    {
        free (unmappedVertexIndices);
    }
    
    if (skinVertexIndices != NULL)
    {
        free (skinVertexIndices);
    }

    if (skinMatrixWeights != NULL)
    {
        free (skinMatrixWeights);
    }

    if (vertexMap != NULL)
    {
        free (vertexMap);
    }

    if (mayaBlindData != NULL)
    {
        delete mayaBlindData;
    }
}

sceneNode & sceneNode::operator*=(sceneNode &b)
{
    RwMatrix matrix;

    /* Ensure animation data is up to date */    
    sampleMayaMatrices();
    b.sampleMayaMatrices();

    /* Multiply the animations */
    animKeys *= b.animKeys;
    
    /* Multiply the base matrices */
    RwMatrixMultiply(&matrix, RwFrameGetMatrix(frame), RwFrameGetMatrix(b.frame));
    RwMatrixCopy(RwFrameGetMatrix(frame), &matrix);

    return *this;
}

void sceneNode::calculateNodeType()
{
    if (numPolys > 0)
    {
        type = geometryNode;
    }
    else if (hasSkinInfluences == true)
    {
        type = jointNode;
    }
    else
    {
        type = unknownNode;
    }
}

void sceneNode::calculatePolyCount()
{   
    int     i, group, numGroups, numFaceIndices;    
    long    *faceIndices;

    numGroups = DtGroupGetCount(mdtIndex);

    for (group = 0; group < numGroups; group++)
    {
        DtFaceGetIndexByShape(mdtIndex, group, &numFaceIndices, &faceIndices);

        for (i = 0; i < numFaceIndices; )
        {
            i += 2;
            while (faceIndices[i] != -1)
            {
                numPolys++;
                i++;
            }
            i++;
        }
    }
}

void sceneNode::checkSkinning()
{
    unsigned int        i, length;
    _skinClusterList    *temp = globalData->gSkinClusterList;
    
    while (temp)
    {
        length = temp->joints->length();
        for (i = 0; i < length; i++)
        {
            if ((*(temp->joints))[i] == dagPath)
            {
                hasSkinInfluences = true;
            }
        }

        if (*temp->skin == dagPath)
        {
            isSkinned = true;
        }

        temp = temp->next;
    }
}

void sceneNode::computeRwMatrix(RwMatrix *matrix, bool clearRoot)
{
    float   *floatArray;
    float   matrixArray[4][4];
    RwV3d   rotPos;
    MMatrix parentMat, shapeLTM, shapeMatrix;   
    RwV3d   parentScale;
    
    DtShapeGetLTM(mdtIndex, &floatArray);
    memcpy(matrixArray, floatArray, sizeof(float) * 16);
    DtShapeGetRotationPivot(mdtIndex, &rotPos.x, &rotPos.y, &rotPos.z);

    matrixArray[3][0] = rotPos.x;
    matrixArray[3][1] = rotPos.y;
    matrixArray[3][2] = rotPos.z;

    /* Get an object offset in local space */
    DtShapeGetLocalRotationPivot(mdtIndex, &rotPos.x, &rotPos.y, &rotPos.z);
    objectOffset.x = - rotPos.x;
    objectOffset.y = - rotPos.y;
    objectOffset.z = - rotPos.z;

    objectScale.x = RwV3dLength((RwV3d *)&matrixArray[0]);
    objectScale.y = RwV3dLength((RwV3d *)&matrixArray[1]);
    objectScale.z = RwV3dLength((RwV3d *)&matrixArray[2]);

    shapeLTM = MMatrix(matrixArray);

    if (DtShapeGetParentID(mdtIndex) != -1)
    {
        /* Get the parents LTM */
        DtShapeGetLTM(DtShapeGetParentID(mdtIndex), &floatArray);
        memcpy(matrixArray, floatArray, sizeof(float) * 16);

        DtShapeGetRotationPivot(DtShapeGetParentID(mdtIndex), &rotPos.x, &rotPos.y, &rotPos.z);
        matrixArray[3][0] = rotPos.x;
        matrixArray[3][1] = rotPos.y;
        matrixArray[3][2] = rotPos.z;

        /* Get the parents scale factors */
        parentScale.x = RwV3dLength((RwV3d *)&matrixArray[0]);
        parentScale.y = RwV3dLength((RwV3d *)&matrixArray[1]);
        parentScale.z = RwV3dLength((RwV3d *)&matrixArray[2]);

        parentMat = MMatrix(matrixArray);
        shapeMatrix = shapeLTM * parentMat.inverse();

        shapeMatrix.get(matrixArray);

        memcpy(&matrix->right, &matrixArray[0], sizeof(RwV3d));
        memcpy(&matrix->up, &matrixArray[1], sizeof(RwV3d));
        memcpy(&matrix->at, &matrixArray[2], sizeof(RwV3d));
        memcpy(&matrix->pos, &matrixArray[3], sizeof(RwV3d));

        matrix->pos.x *= parentScale.x;
        matrix->pos.y *= parentScale.y;
        matrix->pos.z *= parentScale.z;
    }
    else if (clearRoot)
    {
        RwMatrixSetIdentity(matrix);
    }
    else
    {
        shapeLTM.get(matrixArray);

        memcpy(&matrix->right, &matrixArray[0], sizeof(RwV3d));
        memcpy(&matrix->up, &matrixArray[1], sizeof(RwV3d));
        memcpy(&matrix->at, &matrixArray[2], sizeof(RwV3d));
        memcpy(&matrix->pos, &matrixArray[3], sizeof(RwV3d));
    }
    
    /* Remove the scaling it's set in objectScale */
    RwV3dNormalize(&matrix->right, &matrix->right);
    RwV3dNormalize(&matrix->up, &matrix->up);
    RwV3dNormalize(&matrix->at, &matrix->at);

    /* Scale the matrix position by user scale factor */
    RwV3dScale(&matrix->pos, &matrix->pos, globalData->m_scaleFactor);

    RwMatrixUpdate(matrix);
}

void sceneNode::computeSkinnedRwMatrix(RwMatrix *matrix)
{
    RwV3d       rotPos;
    float       *floatArray;
    float       matrixArray[4][4];
    MMatrix     shapeLTM;
    MMatrix     hierarchyRootLTM;

    /* Get the LTM of this node */
    DtShapeGetLTM(mdtIndex, &floatArray);
    memcpy(matrixArray, floatArray, sizeof(float) * 16);
    shapeLTM = MMatrix(matrixArray);

    /* Get the LTM of the root of the skeleton hierarchy */
    DtShapeGetLTM(globalData->hierarchyRootShape, &floatArray);
    memcpy(matrixArray, floatArray, sizeof(float) * 16);
    DtShapeGetRotationPivot(globalData->hierarchyRootShape, &rotPos.x, &rotPos.y, &rotPos.z);

    matrixArray[3][0] = rotPos.x;
    matrixArray[3][1] = rotPos.y;
    matrixArray[3][2] = rotPos.z;
    
    /* Remove the scaling from the root matrix */
    RwV3dNormalize((RwV3d *)(matrixArray[0]), (RwV3d *)(matrixArray[0]));
    RwV3dNormalize((RwV3d *)(matrixArray[1]), (RwV3d *)(matrixArray[1]));
    RwV3dNormalize((RwV3d *)(matrixArray[2]), (RwV3d *)(matrixArray[2]));

    hierarchyRootLTM = MMatrix(matrixArray);

    /* Compute the difference between the two */
    shapeLTM = shapeLTM * hierarchyRootLTM.inverse();
    shapeLTM.get(matrixArray);

    memcpy(&matrix->right, &matrixArray[0], sizeof(RwV3d));
    memcpy(&matrix->up, &matrixArray[1], sizeof(RwV3d));
    memcpy(&matrix->at, &matrixArray[2], sizeof(RwV3d));
    memcpy(&matrix->pos, &matrixArray[3], sizeof(RwV3d));

    /* Scale the matrix position by user scale factor */
    RwV3dScale(&matrix->pos, &matrix->pos, globalData->m_scaleFactor);

    RwMatrixUpdate(matrix);
}

void sceneNode::sampleMayaMatrices()
{
    animKeyIt key, tempKey;
    float time;

    /* Run through the anim key times and sample the Maya modelling matrix at each */
    for (key = animKeys.keys.begin(); key != animKeys.keys.end(); key++)
    {
        time = key->time;

        if (key->dirty == true)
        {
            MAnimControl::setCurrentTime(MTime(key->time, MTime::uiUnit()));

            computeRwMatrix(&key->matrix, false);

            /*
                If this is the root object then make the animation relative to the
                first frame.
            */
            if (DtShapeGetParentID(mdtIndex) == -1)
            {
                RwMatrix *start = RwMatrixCreate();
                RwMatrix *invStart = RwMatrixCreate();
            
                MAnimControl::setCurrentTime(MTime(globalData->m_animStart, MTime::uiUnit()));
            
                computeRwMatrix(start, false);
                RwMatrixInvert(invStart, start);
                RwMatrixCopy(start, &key->matrix);
                RwMatrixMultiply(&key->matrix, start, invStart);
            
                RwMatrixDestroy(start);
                RwMatrixDestroy(invStart);
            }

            RpAnimQuatConvertFromMatrix(&key->quat, &key->matrix);
            key->dirty = false;
        }
    }
}

void sceneNode::computeAnimation()
{
    if (globalData->m_verbose)
    {
        printf("Computing animation on node\n");
    }

    /* Store animation keys based on the Maya keyframes */
    animKeys.computeAnimation(mdtIndex, dagPath);

    sampleMayaMatrices();

    /*
        If requested run through the animation keys and check if we should
        promote any to keyframes to minimise interpolation errors.
    */
    if (globalData->m_dynamicKeyframeGeneration && (animKeys.keys.size() > 0))
    {
        animKeys.checkKeyFrameErrors(globalData->m_keyframeGenerationTolerance);
    }
}

void sceneNode::checkTexturing()
{
    int group, numGroups;
    
    numGroups = DtGroupGetCount(mdtIndex);
    
    for (group = 0; group < numGroups; group++)
    {
        if (MayaShadingGroupIsTextured(mdtIndex, group))
        {
            isTextured = true;
        }
    }
}

void sceneNode::calculateMorphKeys()
{
    if (globalData->m_morphTargets)
    {
        if (globalData->m_morphSampleNotKeys)
        {
            int numKeys = globalData->m_animLength / globalData->m_morphTargetInterval + 1;
            int key;
            
            for (key = 0; key < numKeys; key++)
            {
                morphTargetKeys.append(key * globalData->m_morphTargetInterval);
            }
        }
        else
        {
            DtShapeGetVtxAnimKeys(mdtIndex, &morphTargetKeys);
        }
    
        /* Ensure we have at least a single morph target */
        if (morphTargetKeys.length() < 1)
        {
            morphTargetKeys.append(globalData->m_animStart);
        }
    }
    else
    {
        morphTargetKeys.append(globalData->m_animStart);
    }
}

void sceneNode::calculateRemappedGeometry()
{
    ReMapper    reMapper;    
    int         inputVertCount;
    RwV3d       **uvArrays = NULL;
    DtVec3f     *mayaVerts;
    DtVec3f     *mayaNormals;
    int         i, count, group, numGroups;
    unsigned int        target;
    ReMapperFaceLoop    *remapFace;

    if (numPolys == 0)
    {
        return;
    }

    DtShapeGetVertices(mdtIndex, &inputVertCount, &mayaVerts);
    if (DtShapeIsFlatShaded(mdtIndex))
    {
        DtShapeGetPolygonNormals(mdtIndex, &count, &mayaNormals);
    }
    else
    {
        DtShapeGetNormals(mdtIndex, &count, &mayaNormals);
    }

    numGroups = DtGroupGetCount(mdtIndex);

    if (globalData->m_verbose)
    {
        printf("Remapping input data\n");
        
        printf("Shape has %d input vertices, %d input faces, %d shading groups\n",
               inputVertCount, numPolys, numGroups);
    }
    
    reMapper.SetNumInputFaces(numPolys);

    /* Calculate arrays of transformed uv's ready to remap */
    uvArrays = (RwV3d **) RwMalloc(sizeof(RwV3d *) * numGroups);
    
    for (group = 0; group < numGroups; group++)
    {
        if (isTextured)
        {
            RwMatrix *trans;
            DtVec2f *tVerts;
            
            DtGroupGetTextureVertices(mdtIndex, group, &count, &tVerts);
            
            if (count > 0)
            {
                uvArrays[group] = (RwV3d *) RwMalloc(sizeof(RwV3d) * count);
                
                trans = getUVTransformMatrix(mdtIndex, group);
                
                for (i = 0; i < count; i++)
                {
                    uvArrays[group][i].x = tVerts[i].vec[0];
                    uvArrays[group][i].y = tVerts[i].vec[1];
                    uvArrays[group][i].z = 0.0f;
                }
                
                free(tVerts);
                
                RwV3dTransformPoints(uvArrays[group], uvArrays[group], count, trans);
                
                /* Offset the texture coordinates to positive space and invert the Vs. */
                OffsetTextureCoords(uvArrays[group], count);

                RwMatrixDestroy(trans);
            }
            else
            {
                uvArrays[group] = NULL;
            }
        }
        else
        {
            uvArrays[group] = NULL;
        }
    }

    /* Add all the groups into the remapper */
    for (group = 0; group < numGroups; group++)
    {                        
        reMapper.AddGroup(mdtIndex, group, globalData->materialMap, uvArrays[group],
            globalData->m_preLight, globalData->m_limitUVs, globalData->m_uvLimit);
    }

    /* Remap it all */
    reMapper.DoRemap();

    numRemappedVerts = reMapper.GetNumVertices();
    numRemappedFaces = reMapper.GetNumFaces();

    assert (numRemappedVerts > 0);
    assert (numRemappedFaces > 0);
    
    numRemappedVertsAfterMerge = numRemappedVerts;
    numRemappedFacesAfterMerge = numRemappedFaces;

    if (globalData->m_verbose)
    {
        printf("Remapped to %d faces, %d vertices\n", numRemappedFaces, numRemappedVerts);
    }

    /*
        Store the original vertex indices before the remapping. We will need these
        later when calculating skinning.
    */
    unmappedVertexIndices = (int *) RwMalloc(sizeof(int) * numRemappedVerts);
    
    for (i = 0; i < numRemappedVerts; i++)
    {
        ReMapperVertex *remapVert = reMapper.GetVertex(i);

        unmappedVertexIndices[i] = remapVert->vertexIndex;
    }

    if (globalData->m_preLight)
    {
        remappedPreLightColors = (RwRGBA *) RwMalloc(sizeof(RwRGBA) * numRemappedVerts);
     
        for (i = 0; i < numRemappedVerts; i++)
        {
            ReMapperVertex *remapVert = reMapper.GetVertex(i);

            remappedPreLightColors[i].red   = remapVert->r;
            remappedPreLightColors[i].green = remapVert->g;
            remappedPreLightColors[i].blue  = remapVert->b;
            remappedPreLightColors[i].alpha = remapVert->a;
        }
    }

    if (isTextured)
    {
        remappedTexCoords = (RwTexCoords *) RwMalloc(sizeof(RwTexCoords) * numRemappedVerts);

        for (i = 0; i < numRemappedVerts; i++)
        {
            ReMapperVertex *remapVert = reMapper.GetVertex(i);

            remappedTexCoords[i].u = remapVert->u;
            remappedTexCoords[i].v = remapVert->v;
        }
    }


    if (morphTargetKeys.length() > 1)
    {
        if (isSkinned)
        {
            /* Disable the skin cluster effect so that it doesn't affect morph target data */
            setSkinClusterEnvelopes(0.0f);
        }
    }
    
    remappedPositions   = (RwV3d **) RwMalloc(sizeof(RwV3d *) * morphTargetKeys.length());
    if (globalData->m_exportNormals && mayaNormals)
    {
        remappedNormals = (RwV3d **) RwMalloc(sizeof(RwV3d *) * morphTargetKeys.length());
    }

    for (target = 0; target < morphTargetKeys.length(); target++)
    {
        remappedPositions[target] = (RwV3d *) RwMalloc(sizeof(RwV3d) * numRemappedVerts);

        if (globalData->m_exportNormals && mayaNormals)
        {
            remappedNormals[target] = (RwV3d *) RwMalloc(sizeof(RwV3d) * numRemappedVerts);
        }

        /*
            Set the animation frame to the morph target key. We must use DtFrameSet here
            to ensure the cached MDt geometry data is updated.
        */
        DtFrameSet(morphTargetKeys[target]);

        for (i = 0; i < numRemappedVerts; i++)
        {
            ReMapperVertex *remapVert = reMapper.GetVertex(i);
                      
            remappedPositions[target][i].x = mayaVerts[remapVert->vertexIndex].vec[0];
            remappedPositions[target][i].y = mayaVerts[remapVert->vertexIndex].vec[1];
            remappedPositions[target][i].z = mayaVerts[remapVert->vertexIndex].vec[2];
        
            if (!isSkinned)
            {
                remappedPositions[target][i].x *= objectScale.x;
                remappedPositions[target][i].y *= objectScale.y;
                remappedPositions[target][i].z *= objectScale.z;
                RwV3dAdd(&remappedPositions[target][i], &remappedPositions[target][i], &objectOffset);
            }
            
            RwV3dScale(&remappedPositions[target][i], &remappedPositions[target][i], globalData->m_scaleFactor);

            if (globalData->m_exportNormals && mayaNormals)
            {
                remappedNormals[target][i].x = mayaNormals[remapVert->normalIndex].vec[0];
                remappedNormals[target][i].y = mayaNormals[remapVert->normalIndex].vec[1];
                remappedNormals[target][i].z = mayaNormals[remapVert->normalIndex].vec[2];
            }
        }
    }

    if (isSkinned && (morphTargetKeys.length() > 1))
    {
        /* Re-enable the skin cluster effect as necessary*/
        setSkinClusterEnvelopes(1.0f);
    }

    /* Set the triangles */
    remappedFaces           = (RwUInt16 *) RwMalloc(sizeof(RwUInt16) * numRemappedFaces * 3);
    remappedFaceMaterials   = (RpMaterial **) RwMalloc(sizeof(RpMaterial *) * numRemappedFaces);

    for (i = 0; i < numRemappedFaces; i++)
    {
        remapFace = reMapper.GetFace(i);   

        remappedFaces[(i * 3) + 0] = (RwUInt16) remapFace->index[0];
        remappedFaces[(i * 3) + 1] = (RwUInt16) remapFace->index[1];
        remappedFaces[(i * 3) + 2] = (RwUInt16) remapFace->index[2];

        remappedFaceMaterials[i] = globalData->materialMap[remapFace->materialIndex].rwMaterial;
    }
    
    /* Remap any blind data associated with this node */
    if (mayaBlindData != NULL)
    {
        mayaBlindData->remap(reMapper);
    }

    /* Destroy the materials and uvArrays we created */
    for (group = 0; group < numGroups; group++)
    {
        if (uvArrays[group])
        {
            RwFree(uvArrays[group]);
        }
    }
    RwFree(uvArrays);
}

void sceneNode::generateRwAtomic()
{
    bool            skinned = false;
    unsigned int    target;
    RwInt32         geomFlags = 0;
    RwSphere        boundingSphere;
    float           time, prevTime;
    int             i, mergeNode;
    sceneNode       *node;
    int             mergeVertexOffset, mergeFaceOffset;
    RwMatrix        transform;

    if ((numRemappedVertsAfterMerge == 0) || (numRemappedFacesAfterMerge == 0))
    {
        return;
    }

    /* Generate an atomic */
    atomic = RpAtomicCreate();    

    /* Now build a geometry and set the data */
    if (remappedNormals != NULL)
    {
        geomFlags |= rpGEOMETRYNORMALS;
    }
    if (isLit)
    {
        geomFlags |= rpGEOMETRYLIGHT;
    }
    if (remappedTexCoords != NULL)
    {
        geomFlags |= rpGEOMETRYTEXTURED;
    }
    if (remappedPreLightColors != NULL)
    {
        geomFlags |= rpGEOMETRYPRELIT;
    }
    if (globalData->m_triStrips)
    {
        geomFlags |= rpGEOMETRYTRISTRIP;
    }
    
    RpGeometry *geom = RpGeometryCreate(numRemappedVertsAfterMerge,
                        numRemappedFacesAfterMerge, geomFlags);

    if (morphTargetKeys.length() > 1)
    {
        if (globalData->m_verbose)
        {
            printf("Adding %d morph targets\n", morphTargetKeys.length());
        }

        RpMorphGeometryCreateInterpolators(geom, morphTargetKeys.length() - 1);
        
        for (target = 0; target < morphTargetKeys.length(); target++)
        {
            time = ((float)MTime(morphTargetKeys[target], MTime::uiUnit()).as(MTime::kSeconds)) -
                   ((float)MTime(globalData->m_animStart, MTime::uiUnit()).as(MTime::kSeconds));

            if (target > 0)
            {
                RpGeometryAddMorphTarget(geom);
                RpMorphGeometrySetInterpolator(geom, target - 1, target - 1, target, time - prevTime);
            }

            prevTime = time;
        }

    }

    /* First add all the triangles to the geometry */
    mergeVertexOffset = mergeFaceOffset = 0;

    for (mergeNode = 0; mergeNode < mergedNodes.size(); mergeNode++)
    {
        node = mergedNodes[mergeNode];

        /* Set the triangles */
        for (i = 0; i < node->numRemappedFaces; i++)
        {
            geom->triangles[mergeFaceOffset + i].vertIndex[0] = mergeVertexOffset + node->remappedFaces[(i * 3) + 0];
            geom->triangles[mergeFaceOffset + i].vertIndex[1] = mergeVertexOffset + node->remappedFaces[(i * 3) + 1];
            geom->triangles[mergeFaceOffset + i].vertIndex[2] = mergeVertexOffset + node->remappedFaces[(i * 3) + 2];

            RpGeometryTriangleSetMaterial(geom, &geom->triangles[mergeFaceOffset + i], node->remappedFaceMaterials[i]);
        }

        mergeVertexOffset   += node->numRemappedVerts;
        mergeFaceOffset     += node->numRemappedFaces;
    }

    /* Construct a vertex map that will sort the geometry vertices by material */
    vertexMap = calculateVertexMap(geom);

    /* Add the per-vertex data to the geometry. */
    mergeVertexOffset = 0;
    
    for (mergeNode = 0; mergeNode < mergedNodes.size(); mergeNode++)
    {
        node = mergedNodes[mergeNode];

        for (target = 0; target < node->morphTargetKeys.length(); target++)
        {
            RwV3d           *verts;
            RwV3d           *norms;
            RpMorphTarget   *morphtarget;

            morphtarget = RpGeometryGetMorphTarget(geom, target);
            verts       = RpMorphTargetGetVertices(morphtarget);

            if (node->remappedNormals != NULL)
            {
                norms = RpMorphTargetGetVertexNormals(morphtarget);
            }

            calculateOffsetMatrix(&transform, RwFrameGetLTM(frame), RwFrameGetLTM(node->frame));

            for (i = 0; i < node->numRemappedVerts; i++)
            {
                RwV3dTransformPoints(&verts[vertexMap[mergeVertexOffset + i]],
                    &node->remappedPositions[target][i], 1,
                    &transform);

                if (node->remappedNormals != NULL)
                {
                    norms[vertexMap[mergeVertexOffset + i]].x = node->remappedNormals[target][i].x;
                    norms[vertexMap[mergeVertexOffset + i]].y = node->remappedNormals[target][i].y;
                    norms[vertexMap[mergeVertexOffset + i]].z = node->remappedNormals[target][i].z;
                }
            }
            

            RpMorphTargetCalcBoundingSphere(morphtarget, &boundingSphere);
            RpMorphTargetSetBoundingSphere(morphtarget, &boundingSphere);
        }

        if (node->remappedTexCoords != NULL)
        {
            RwTexCoords *texCoords = RpGeometryGetVertexTexCoords(geom, rwTEXTURECOORDINATEINDEX0);

            for (i = 0; i < node->numRemappedVerts; i++)
            {
                texCoords[vertexMap[mergeVertexOffset + i]].u = node->remappedTexCoords[i].u;
                texCoords[vertexMap[mergeVertexOffset + i]].v = node->remappedTexCoords[i].v;
            }

        }

        if (node->remappedPreLightColors != NULL)
        {
            RwRGBA *preLightColors = RpGeometryGetPreLightColors(geom);

            for (i = 0; i < node->numRemappedVerts; i++)
            {
                preLightColors[vertexMap[mergeVertexOffset + i]].red     = node->remappedPreLightColors[i].red;
                preLightColors[vertexMap[mergeVertexOffset + i]].green   = node->remappedPreLightColors[i].green;
                preLightColors[vertexMap[mergeVertexOffset + i]].blue    = node->remappedPreLightColors[i].blue;
                preLightColors[vertexMap[mergeVertexOffset + i]].alpha   = node->remappedPreLightColors[i].alpha;
            }
        }

        mergeVertexOffset   += node->numRemappedVerts;
    }

    /* Merge together all the blind data on the nodes */
    for (mergeNode = 1; mergeNode < mergedNodes.size(); mergeNode++)
    {
        node = mergedNodes[mergeNode];

        *mayaBlindData += *(node->mayaBlindData);
    }

    /* Apply our vertex map to the blind data */
    mayaBlindData->applyMap(vertexBlindDataType, vertexMap, true);

    /* Add the blind data to the geometry */
    mayaBlindData->addToGeometry(geom);

    RpGeometryUnlock(geom);

    if (GeometryHasColoredMaterials(geom))
    {
        RpGeometrySetFlags(geom, RpGeometryGetFlags(geom) | rpGEOMETRYMODULATEMATERIALCOLOR);
    }
    
    if (GeometryHasMaterialFX(geom))
    {
        RpMatFXAtomicEnableEffects(atomic);
    }

    RpAtomicSetGeometry(atomic, geom, 0);

    /* Reduce the ref count so it's destroyed with the atomic */
    RpGeometryDestroy(geom);

    if (isSkinned)
    {
        buildSkinVertexMapping();
    }
}

void sceneNode::generateSkinnedRwAtomic(RpClump *clump, unsigned int numNodes,
                                        RwMatrix *invMatrices, RwUInt32 *animFlags,
                                        RwInt32 *tags)
{
    RpGeometry  *geometry;

    /*
        Set the animation frame to animation start. We must use DtFrameSet here
        to ensure the cached MDt geometry data is updated.
    */
    DtFrameSet(globalData->m_animStart);

    /* Create an atomic */
    generateRwAtomic();

    /*
        Attach the atomic to the root of the hierarchy and transform the vertices
        as necessary.
    */
    geometry = RpAtomicGetGeometry(atomic);
    RpGeometryTransform(geometry, RwFrameGetMatrix(frame));

    RpAtomicSetFrame(atomic, RpClumpGetFrame(clump));
    RpClumpAddAtomic(clump, atomic);

    /* Create skin data for the atomic */
    RpSkin *skin = RpSkinCreate(atomic, numNodes, skinMatrixWeights,
                                skinVertexIndices, invMatrices,
                                animFlags);
    if (skin)
    {
        RwUInt32 i;
        /* set the bonetags to match the frame hierarchy */
        for (i = 0; i < numNodes; i++)
        {
            skin->pBoneInfo[i].boneTag = tags[i];
        }
    }
}

void
sceneNode::buildSkinVertexMapping()
{
    int         i, mergeNode;
    MObject     shapeObj;
    MDagPath    shapeDAG;
    sceneNode   *node;
    int         mergeVertexOffset = 0;

    if ((numRemappedVertsAfterMerge == 0) ||
        (numRemappedFacesAfterMerge == 0))
    {
        return;
    }

    skinVertexIndices = (RwUInt32 *)malloc(sizeof(RwUInt32) * numRemappedVertsAfterMerge);
    skinMatrixWeights = (RwMatrixWeights *)malloc(sizeof(RwMatrixWeights) * numRemappedVertsAfterMerge);
    
    for (mergeNode = 0; mergeNode < mergedNodes.size(); mergeNode++)
    {
        node = mergedNodes[mergeNode];

        DtExt_ShapeGetShapeNode(node->mdtIndex, shapeObj);
        shapeDAG.getAPathTo(shapeObj, shapeDAG);

        for (i = 0; i < node->numRemappedVerts; i++)
        {        
            int numWeightsSet = 0;

            skinVertexIndices[vertexMap[mergeVertexOffset + i]]    = 0;
            skinMatrixWeights[vertexMap[mergeVertexOffset + i]].w0 = 0.0f;
            skinMatrixWeights[vertexMap[mergeVertexOffset + i]].w1 = 0.0f;
            skinMatrixWeights[vertexMap[mergeVertexOffset + i]].w2 = 0.0f;
            skinMatrixWeights[vertexMap[mergeVertexOffset + i]].w3 = 0.0f;
        
            /*
               Ok we know the vert now loop through all the skinClusters and for
               each affecting this geometry get the weights on this vert, and match
               to our list of bones.
            */
            _skinClusterList *temp = globalData->gSkinClusterList;
            while (temp)
            {
                if (shapeDAG == *temp->skin)
                {                
                    /* process this cluster for this skin */
                    MStatus stat;
                    MDagPathArray infs;
                    unsigned int nInfs = temp->skinCluster->influenceObjects(infs, &stat);
            
                    /* create an iterator for this geometry */
                    MItGeometry gIter(*temp->skin);

                    for (/* nothing */ ; !gIter.isDone(); gIter.next()) 
                    {
                        if (gIter.index() == node->unmappedVertexIndices[i])
                        {
                            /* find the entry for this vert and get it's component */
                            MObject comp = gIter.component(&stat);

                            /* Get the weights for this vertex (one per influence object) */
                            MFloatArray wts;
                            unsigned int infCount;
                            stat = temp->skinCluster->getWeights(shapeDAG, comp, wts, infCount);

                            /* Output the weight data for this vertex */
                            for (unsigned int jj = 0; jj < infCount ; jj++) 
                            {
                                RwReal weight = wts[jj];
                                MDagPath *bone = &infs[jj];

                                if (wts[jj] > 0.0f)
                                {
                                    RwInt32 boneID = -1;
                                    RwInt32 index, minWeightIndex;
                                    RwReal  minWeight;
                                    void *tempPtr = (void *)(&skinMatrixWeights[vertexMap[mergeVertexOffset + i]]);
                                    RwReal *weightArrayPtr = (RwReal *)tempPtr;

                                    /* find the bone ID of this influence obj */
                                    _dagPathBoneIndexList *tempDag = globalData->gDagPathBoneIndexList;
                                    while (tempDag && boneID == -1)
                                    {
                                        if (*tempDag->object == infs[jj])
                                        {
                                            boneID = tempDag->index;
                                        }
                                        tempDag = tempDag->next;
                                    }

                                    if (boneID == -1)
                                    {
                                        continue;
                                    }

                                    /* Find the lowest set weight */
                                    minWeightIndex = 0;
                                    minWeight = weightArrayPtr[0];

                                    for (index = 1; index < 4; index++)
                                    {
                                        if (weightArrayPtr[index] < minWeight)
                                        {
                                            minWeightIndex = index;
                                            minWeight = weightArrayPtr[index];
                                        }
                                    }

                                    if (wts[jj] > minWeight)
                                    {
                                        weightArrayPtr[minWeightIndex] = wts[jj];
                                        skinVertexIndices[vertexMap[mergeVertexOffset + i]] &= ~(0xFF << (minWeightIndex * 8));
                                        skinVertexIndices[vertexMap[mergeVertexOffset + i]] |= boneID << (minWeightIndex * 8);
                                    }
                                }
                            }
                        }
                    }
                }
                temp = temp->next;
            }
        }
        mergeVertexOffset   += node->numRemappedVerts;
    }
}

/* General Functions */
void calculateOffsetMatrix(RwMatrix *out, RwMatrix *from, RwMatrix *to)
{
    RwMatrix inverseFrom;

    RwMatrixInvert(&inverseFrom, from);

    RwMatrixMultiply(out, &inverseFrom, to);
}


int setSkinClusterEnvelopes(float envelope)
{
    _skinClusterList *temp = globalData->gSkinClusterList;

    while (temp)
    {
        temp->skinCluster->setEnvelope(envelope);
        temp = temp->next;
    }

    return 1;
}

int DtShapeGetLTM(int shapeID, float **matrix)
{
    static float	mtxI[4][4];
    static float    mtxInclusive[4][4];
    static float    mtxLocal[4][4];

    /* Initialize return values. */
    mtxI[0][0] = 1.0; mtxI[0][1] = 0.0; mtxI[0][2] = 0.0; mtxI[0][3] = 0.0;
    mtxI[1][0] = 0.0; mtxI[1][1] = 1.0; mtxI[1][2] = 0.0; mtxI[1][3] = 0.0;
    mtxI[2][0] = 0.0; mtxI[2][1] = 0.0; mtxI[2][2] = 1.0; mtxI[2][3] = 0.0;
    mtxI[3][0] = 0.0; mtxI[3][1] = 0.0; mtxI[3][2] = 0.0; mtxI[3][3] = 1.0;

    *matrix = (float *)&mtxI;
    
    if (shapeID == -1)
    {
        return (-1);
    }

    /* Get the Transform from the object. */
    MStatus stat = MS::kSuccess;
    MObject transformNode;

    DtExt_ShapeGetTransform(shapeID, transformNode);

    /* Take the first dag path. */
    MFnDagNode fnTransNode(transformNode, &stat);
    MDagPath dagPath;
    stat = fnTransNode.getPath(dagPath);

    MFnDagNode fnDagPath(dagPath, &stat);

    MMatrix mayaMatrix;
    MMatrix localMatrix;
    MMatrix testMatrix;

    mayaMatrix = dagPath.inclusiveMatrix();
    testMatrix = dagPath.exclusiveMatrix();
    localMatrix = fnTransNode.transformationMatrix(&stat);

    mayaMatrix.get(mtxInclusive);
    *matrix = (float *)&mtxInclusive;

    return(1);

}  

int DtShapeGetLocalRotationPivot(int shapeID, float *x, float *y, float *z)
{
    
    MObject transformNode;
    /* Get the Transform from the object. */
    DtExt_ShapeGetTransform(shapeID, transformNode);
    
    /* Take the first dag path. */
    MFnDagNode fnTransNode(transformNode);
    MDagPath dagPath;
    fnTransNode.getPath(dagPath);
    
    MFnDagNode fnDagPath(dagPath);

    MFnTransform transFn(dagPath);

    /* Get the pivot in local space */
    MPoint rP = transFn.rotatePivot(MSpace::kTransform);

    *x = (float)rP.x;
    *y = (float)rP.y;
    *z = (float)rP.z;

	return 1;
}


RpMaterial *buildVertexMap(RpMaterial *material, void *pData)
{
    GeomSort    *gs = (GeomSort *)pData;
    RwInt32     numTriangles = RpGeometryGetNumTriangles(gs->geom);
    RpTriangle  *tris = RpGeometryGetTriangles(gs->geom);
    RwInt32     i, j;

    for (i = 0; i<numTriangles; i++)
    {
        if (RpGeometryGetMaterial(gs->geom, tris[i].matIndex) == material)
        {
            for (j = 0; j < 3; j++)
            {
                if (gs->vertexMap[tris[i].vertIndex[j]] == -1)
                {
                    gs->vertexMap[tris[i].vertIndex[j]] = gs->currentIndex;
                    gs->currentIndex++;
                }
            }
        }
    }

    return (material);
}

int *calculateVertexMap(RpGeometry *geometry)
{
    GeomSort    gs;
    int         i, j;
    int         numVertices, numTris;
    int         *vertexMap;
    RpTriangle  *tris;

    numTris = RpGeometryGetNumTriangles(geometry);
    numVertices = RpGeometryGetNumVertices(geometry);
    
    vertexMap = (int *)RwMalloc(sizeof(RwInt32) * numVertices);

    for (i = 0; i < numVertices; i++)
    {
        vertexMap[i] = -1;
    }

    gs.geom         = geometry;
    gs.vertexMap    = vertexMap;
    gs.currentIndex = 0;

    RpGeometryForAllMaterials(geometry, buildVertexMap, &gs);

    tris = RpGeometryGetTriangles(geometry);

    for (i = 0; i < numTris; i++)
    {
        for (j = 0; j < 3; j++)
        {
            tris[i].vertIndex[j] = (RwUInt16)vertexMap[tris[i].vertIndex[j]];
        }
    }

    return(vertexMap);
}

